
/*
  CLASSiC DAC, Copyright 2013 SILICON CHIP Publications
  spi.c: SD/MMC card SPI communication functions
  Written by Mauro Grassi and updated by Nicholas Vinen, 2009-2013
*/

//// change these to suit the application ////

#define SDCARD_SPI        1
#define SDCARD_DMA		  1
#define SDCARD_WDMA		  2
#define SDCARD_DIN_RP     2
#define SDCARD_DIN_PORT   F
#define SDCARD_DIN_PIN    2
#define SDCARD_CLK_RP     6
#define SDCARD_CLK_PORT   F
#define SDCARD_CLK_PIN    6
#define SDCARD_DOUT_RP    3
#define SDCARD_DOUT_PORT  F
#define SDCARD_DOUT_PIN   3

/////////////////////////////////////////////

#include "GenericTypeDefs.h"
#include "spi.h"
#include "p33Fxxxx.h"
#include <string.h>

/* 
	SPI Module Firmware
	by Mauro Grassi, 2009-2010.
*/

#define _SPISTATname(spi) SPI##spi##STAT
#define SPISTATname(spi)  _SPISTATname(spi)
#define _SPISTATbits(spi) SPI##spi##STATbits
#define SPISTATbits(spi)  _SPISTATbits(spi)
#define _SPICON1name(spi) SPI##spi##CON1
#define SPICON1name(spi)  _SPICON1name(spi)
#define _SPICON1bits(spi) SPI##spi##CON1bits
#define SPICON1bits(spi)  _SPICON1bits(spi)
#define _SPICON2name(spi) SPI##spi##CON2
#define SPICON2name(spi)  _SPICON2name(spi)
#define _SPIBUFname(spi)  SPI##spi##BUF
#define SPIBUFname(spi)   _SPIBUFname(spi)
#define _SPIIEname(spi)   _SPI##spi##IE
#define SPIIEname(spi)    _SPIIEname(spi)
#define _SPIIFname(spi)   _SPI##spi##IF
#define SPIIFname(spi)    _SPIIFname(spi)
#define _RPORbits(a,b)    RPOR##a##bits.RP##b##R
#define RPORbits(a,b)     _RPORbits(a,b)
#define _TRISbits(a,b)    TRIS##a##bits.TRIS##a##b
#define TRISbits(a,b)     _TRISbits(a,b)
#define _LATbits(a,b)     LAT##a##bits.LAT##a##b
#define LATbits(a,b)      _LATbits(a,b)
#define _DMACONname(dma)  DMA##dma##CON
#define DMACONname(dma)   _DMACONname(dma)
#define _DMACONbits(dma)  DMA##dma##CONbits
#define DMACONbits(dma)   _DMACONbits(dma)
#define _DMASTAname(dma)  DMA##dma##STA
#define DMASTAname(dma)   _DMASTAname(dma)
#define _DMASTBname(dma)  DMA##dma##STB
#define DMASTBname(dma)   _DMASTBname(dma)
#define _DMAPADname(dma)  DMA##dma##PAD
#define DMAPADname(dma)   _DMAPADname(dma)
#define _DMACNTname(dma)  DMA##dma##CNT
#define DMACNTname(dma)   _DMACNTname(dma)
#define _DMAREQname(dma)  DMA##dma##REQ
#define DMAREQname(dma)   _DMAREQname(dma)

#define SLOW_SPI_DELAY 4096

#ifdef USE_DMA
unsigned char* SPIRxBuf;
volatile unsigned short SPIRxLeft;
#endif

void DeinitSPI() {
	SDCS_TRIS=1;
	SDCS=0;
	SPISTATname(SDCARD_SPI)=0;
	SPICON1name(SDCARD_SPI)=0;
	SPICON2name(SDCARD_SPI)=0;
}

void InitSPI(int speed)
{
	/* Initialise the SPI System */
	SDCS=1;
	SDCS_TRIS=0;
	SPISTATname(SDCARD_SPI)=0;
	SPICON1name(SDCARD_SPI)=0;
	SPICON2name(SDCARD_SPI)=0;

	// set up SPI pins
	TRISbits(SDCARD_DOUT_PORT, SDCARD_DOUT_PIN) = 0;
	LATbits(SDCARD_DOUT_PORT, SDCARD_DOUT_PIN) = 1;
	TRISbits(SDCARD_CLK_PORT, SDCARD_CLK_PIN) = 0;
	TRISbits(SDCARD_DIN_PORT, SDCARD_DIN_PIN) = 1;
	
	/* Speed Selection
		SPIxCON1<4:2>= 	Secondary Prescaler (Master Mode)
						111b	1:1
						110b	2:1
						...	
						000b	8:1
		SPIxCON1<1:0>=	Primary Prescaler (Master Mode)
						11b		1:1
						10b		4:1
						01b		16:1
						00b		64:1

		Note: do not set both to 1:1
	*/

	SPICON1name(SDCARD_SPI)=(0x1F & speed);
	SPICON1bits(SDCARD_SPI).MODE16=0;		/*  8 bit mode */
	SPICON1bits(SDCARD_SPI).SMP=0;			/* input data sampling */
	SPICON1bits(SDCARD_SPI).CKP=0;			/* clock Polarity 0: idle low, active high 1: idle high, active low */
	SPICON1bits(SDCARD_SPI).CKE=1;			/* clock edge selection */
	SPICON1bits(SDCARD_SPI).MSTEN=1;		/* enable Master Mode */
	SPISTATbits(SDCARD_SPI).SPIROV=0;
	SPIIFname(SDCARD_SPI)=0;
	SPIIEname(SDCARD_SPI)=0;
	SPISTATbits(SDCARD_SPI).SPIEN=1;
}

unsigned int WriteSPI(unsigned int x) {
	SDCS=0;
	if( (SPICON1name(SDCARD_SPI)&0x1F) == 0x01 ) {
		unsigned short i;
		for( i = 0; i < SLOW_SPI_DELAY; ++i )
			Nop();

		SPIBUFname(SDCARD_SPI)=x;
		while(SPISTATbits(SDCARD_SPI).SPIRBF==0 && SPISTATbits(SDCARD_SPI).SPIEN)
			;
		x=SPIBUFname(SDCARD_SPI);

		for( i = 0; i < SLOW_SPI_DELAY; ++i )
			Nop();
		SDCS=1;
		for( i = 0; i < SLOW_SPI_DELAY; ++i )
			Nop();
	} else {
		SPIBUFname(SDCARD_SPI)=x;
		while(SPISTATbits(SDCARD_SPI).SPIRBF==0&&SPISTATbits(SDCARD_SPI).SPIEN)
			;
		x=SPIBUFname(SDCARD_SPI);
		SDCS=1;
	}
	return x;
}

#ifdef USE_DMA
void __attribute__((__interrupt__,no_auto_psv)) _SPI1Interrupt(void) {
	register unsigned short temp;

	temp = SPIBUFname(SDCARD_SPI);
    if( --SPIRxLeft )
		SPIBUFname(SDCARD_SPI) = 0;
    ((unsigned short*)SPIRxBuf)[0] = (temp>>8)|(temp<<8);
    SPIRxBuf += 2;
	SPIIFname(SDCARD_SPI)=0;
}

void BulkReadSPI(BYTE* data) {
	SPISTATbits(SDCARD_SPI).SPIEN=0;
	SPICON1bits(SDCARD_SPI).DISSDO=1;
	SPICON1bits(SDCARD_SPI).MODE16=1;
	SPISTATbits(SDCARD_SPI).SPIEN=1;

	SDCS = 0;

	SPIRxBuf = data;
	SPIRxLeft = 256;
	SPIIFname(SDCARD_SPI)=0;
	SPIIEname(SDCARD_SPI)=1;
	SPIBUFname(SDCARD_SPI) = 0;
	while( SPIRxLeft && SPISTATbits(SDCARD_SPI).SPIEN )
		;

	SPISTATbits(SDCARD_SPI).SPIEN=0;
	SPICON1bits(SDCARD_SPI).DISSDO=0;
	SPICON1bits(SDCARD_SPI).MODE16=0;
	SPISTATbits(SDCARD_SPI).SPIEN=1;
	SPIIEname(SDCARD_SPI)=0;

	SDCS = 1;
}
#endif

unsigned int WriteSPIWithoutSS(unsigned int x)
{
	if( (SPICON1name(SDCARD_SPI)&0x1F) == 0x00 ) {
		unsigned short i;
		for( i = 0; i < SLOW_SPI_DELAY; ++i )
			Nop();
	}
	SPIBUFname(SDCARD_SPI)=x;
	while(SPISTATbits(SDCARD_SPI).SPIRBF==0)
		;
	x=SPIBUFname(SDCARD_SPI);
	if( (SPICON1name(SDCARD_SPI)&0x1F) == 0x00 ) {
		unsigned short i;
		for( i = 0; i < SLOW_SPI_DELAY; ++i )
			Nop();
	}
	return x;
}
